## Distributions from age-specific bequest pool by age-specific wealth group
## e.g. total to go to ages 60-65 is $100. Lowest wealth group might get 15%, middle 35%, top 50%

# Prelims -----------------------------------------------------------------

rm(list=ls())
gc()


# Read HILDA --------------------------------------------------------------

hilda_grouped <- qread("./Input data/Intermediate input data/hilda_grouped_master.qs")


# Bequest distributions ---------------------------------------------------

## based on bequest received
age_wlth_beq_dist <- hilda_grouped %>% 
  ## filter to years with wealth data
  filter(wavenumber %in% c(2, 6, 10, 14, 18)) %>% 
  ## filter to responding persons (for bequest question)
  filter(hhwtrp>=0) %>% 
  ## total_wealth as defined in the model
  mutate(total_wealth = total_assets - housing_debt) %>% 
  ## create year and age specific wealth groups
  group_by(wavenumber, age_grp) %>% 
  group_split() %>% 
  lapply(., function(x) {
    x %>% 
      ## for ages <30 and >=90, assume even split of bequest pool (v unlikely to get bequest anyway)
      mutate(age_wlth_grp3 = ifelse(age_grp>="[30,35)" & age_grp<="[85,90)",
                                    cut(round(total_wealth, 2), 
                                        ## cut points determined by weighted quantile (only works if cut points are unique)
                                        breaks = wtd.quantile(.$total_wealth, 
                                                              weights = .$hhwtrp, 
                                                              probs = seq(0, 1, 1/3)) %>% round(2), 
                                        include.lowest=T,
                                        labels = c(1:3),
                                        ordered_result=T),
                                    0), ## 0 indicates it'll be an even split
             ## fix issue with cutting at max or min sometimes
             age_wlth_grp3 = ifelse(is.na(age_wlth_grp3) & round(total_wealth, 2) == round(min(total_wealth), 2) ,
                                    1, age_wlth_grp3) ,
             age_wlth_grp3 = ifelse(is.na(age_wlth_grp3) & round(total_wealth, 2) == round(max(total_wealth), 2) ,
                                    3, age_wlth_grp3)
      )
  }) %>% 
  rbindlist %>% 
  ## prev recorded wealth group as a proxy for pre-bequest wealth group
  group_by(xwaveid) %>% 
  arrange(xwaveid, wavenumber) %>% 
  mutate(prev_age_wlth_grp3 = lag(age_wlth_grp3)) %>% 
  ## remove 0s and NAs -- 0s means they were not within our age ranges for age-specific wealth splits previously, and NA means no previous value
  filter(!is.na(prev_age_wlth_grp3) & prev_age_wlth_grp3!=0) %>% 
  ## calc share of total bequests by age grp received by prev wlth grp
  group_by(wavenumber, age_grp) %>% 
  mutate(age_bequest_tot = sum(bequests*hhwtrp, na.rm=T)) %>% 
  group_by(wavenumber, age_grp, prev_age_wlth_grp3) %>% 
  summarise(age_wlth_bequest_dist = sum(bequests*hhwtrp, na.rm=T)/mean(age_bequest_tot) ) %>% 
  ## take average across waves
  group_by(age_grp, prev_age_wlth_grp3) %>% 
  summarise(age_wlth_bequest_dist = mean(age_wlth_bequest_dist, na.rm=T)) %>% 
  ## take average across age groups as it looks somewhat stable across main receiving age groups (with some bumps)
  group_by(prev_age_wlth_grp3) %>% 
  summarise(age_wlth_bequest_dist = mean(age_wlth_bequest_dist, na.rm=T)) %>% ## sums to 1
  ## add in 0 for age groups <30 or >90 - indicates distributed evenly at these age groups
  rbind(data.frame(prev_age_wlth_grp3 = 0, age_wlth_bequest_dist=1))

# ggplot(age_wlth_beq_dist) + geom_col(aes(x = prev_age_wlth_grp3, y=age_wlth_bequest_dist))

## USED IN APPENDIX
## In the base scenario model, about 13 per cent of inheritances received by the 30–34 to 85–89 age groups was assumed to go 
## to the bottom wealth tertile, 33 per cent to the middle tertile and 54 per cent to the top tertile. 
age_wlth_beq_dist

## save
qsave(age_wlth_beq_dist, "./Input data/age_wlth_beq_dist.qs")

